using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.CodeDom.Compiler;
using Microsoft.CSharp;

namespace BatCompiler
{
    public partial class Compiler : Form
    {
        private bool isdirty = false;
        private string path = "";
        About.AboutBox box = new About.AboutBox("Bat File Compiler");

        public Compiler()
        {
            InitializeComponent();
            this.Icon = Properties.Resources.bat;
            box.programname = "Bat File Compiler";
        }

        private bool save(string path)
        {
            if (string.IsNullOrEmpty(path))
            {
                if (saveFileDialog.ShowDialog() == DialogResult.OK)
                {
                    path = saveFileDialog.FileName;
                }
                else
                    return false;
            }

            if (!string.IsNullOrEmpty(path))
            {
                FileStream str = new FileStream(path, FileMode.Create, FileAccess.Write, FileShare.None);
                using (StreamWriter writer = new StreamWriter(str))
                {
                    writer.Write(this.textBox.Text);
                }
                str.Close();

                this.path = path;
                isdirty = false;
                return true;
            }
            return false;
        }

        private bool check()
        {
            if (isdirty)
            {
                string name = path;
                if (string.IsNullOrEmpty(name))
                {
                    name = "Untitled";
                }

                DialogResult res = MessageBox.Show(name + " has been modified. Do you want to save it?", "File modified", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Information);
                if (res == DialogResult.Yes)
                {
                    return save(path);
                }

                if (res == DialogResult.Cancel)
                {
                    return false;
                }
            }
            return true;
        }

        private bool compile(string path)
        {
            bool result = false;
            string filepath = "";
            if (Properties.Settings.Default.RunHidden)
            {
                filepath = Environment.GetEnvironmentVariable("TEMP") + "hideit.bat";
            }
            else
                filepath = Environment.GetEnvironmentVariable("TEMP") + "it.bat";

            using (StreamWriter sw=new StreamWriter(filepath))
            {
                sw.Write(this.textBox.Text);
            }

            using (CSharpCodeProvider code=new CSharpCodeProvider())
            {
                CompilerParameters compar = new CompilerParameters();

                string option = "/target:winexe";
                if (Properties.Settings.Default.Customicon && File.Exists(Properties.Settings.Default.iconpath))
                {
                    option += " " + "/win32icon:" + "\"" + Properties.Settings.Default.iconpath + "\"";
                }

                compar.CompilerOptions = option;
                compar.GenerateExecutable = true;
                compar.IncludeDebugInformation = false;

                if (File.Exists(filepath))
                {
                    compar.EmbeddedResources.Add(filepath); 
                }

                compar.OutputAssembly = path;
                compar.GenerateInMemory = false; 
                
                compar.ReferencedAssemblies.Add("System.dll");
                compar.ReferencedAssemblies.Add("System.Data.dll");
                compar.ReferencedAssemblies.Add("System.Deployment.dll");
                compar.ReferencedAssemblies.Add("System.Drawing.dll");
                compar.ReferencedAssemblies.Add("System.Windows.Forms.dll");
                compar.ReferencedAssemblies.Add("System.Xml.dll");

                compar.TreatWarningsAsErrors = false;

                CompilerResults res = code.CompileAssemblyFromSource(compar, Properties.Resources.Program);

                if (res.Errors.Count > 0)
                {
                    result = false;
                }
                else
                    result = true;
            }

            return result;
        }

        private void openToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                if (!check()) return;
                this.textBox.Text = File.ReadAllText(openFileDialog.FileName);
                path = openFileDialog.FileName;
                this.textBox.SelectionStart = this.textBox.Text.Length;
                this.Buildstatus.Visible = false;
            }
        }

        private void newToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (!check()) return;

            this.textBox.Text = "";
            this.isdirty = false;
            this.path = "";
            this.Buildstatus.Visible = false;
        }

        private void saveToolStripMenuItem_Click(object sender, EventArgs e)
        {
            save(this.path);
        }

        private void saveAsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            save(null);
        }

        private void textBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            isdirty = true;
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void Compiler_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = !check();
        }

        private void undoToolStripMenuItem_Click(object sender, EventArgs e)
        {
            textBox.Undo();
        }

        private void redoToolStripMenuItem_Click(object sender, EventArgs e)
        {
            textBox.Undo();
            this.textBox.SelectionStart = textBox.Text.Length;
        }

        private void cutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            textBox.Cut();
        }

        private void copyToolStripMenuItem_Click(object sender, EventArgs e)
        {
            textBox.Copy();
        }

        private void pasteToolStripMenuItem_Click(object sender, EventArgs e)
        {
            textBox.Paste();
        }

        private void selectAllToolStripMenuItem_Click(object sender, EventArgs e)
        {
            textBox.SelectAll();
        }

        private void buildToolStripMenuItem_Click(object sender, EventArgs e)
        {
            string path = "";
            if (Properties.Settings.Default.Saveexe && !string.IsNullOrEmpty(this.path))
            {
                path = Path.ChangeExtension(this.path, "exe");
            }
            else
            {
                SaveFileDialog save = new SaveFileDialog();
                save.Filter = "Executable files|*.exe";

                if (save.ShowDialog() == DialogResult.OK)
                {
                    path = save.FileName;
                }
                else
                    return;
            }
            this.StatusProgressBar.Style = ProgressBarStyle.Marquee;
            backgroundWorker1.RunWorkerAsync(path);
        }

        private void optionsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Options opt = new Options();
            opt.ShowDialog();
        }

        private void Compiler_Resize(object sender, EventArgs e)
        {
            this.StatusProgressBar.Width = this.Width - 150;
        }

        private void backgroundWorker1_DoWork(object sender, DoWorkEventArgs e)
        {
            this.UseWaitCursor = true;
            this.Buildstatus.Text = "Building...";
            this.Buildstatus.Visible = true;
            e.Result = compile(e.Argument as string);
        }

        private void backgroundWorker1_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            this.UseWaitCursor = false;
            if ((bool)e.Result)
            {
                this.Buildstatus.Text = "Build Succedded";
            }
            else
                this.Buildstatus.Text = "Build failed";
            this.Buildstatus.Visible = true;
            this.StatusProgressBar.Style = ProgressBarStyle.Blocks;
        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            box.ShowDialog();
        }
    }
}